--[[
	LootDestroyer

	By sarf

	This mod allows loot to be automatically destroyed.
	The loot that should be destroyed must be specified by the user
	(by rightclicking it in the LootFrame).
	
	It destroys the item by picking up the last item it encounters in 
	your inventory that matches the unwanted item name.
	
	Currently, it gives quite a lot of feedback, that will be possible to tune/turn off later on.
	
	Thanks go to Mugendai for his code, which remains an inspiration for me.
	Thanks also goes to Bayta, who came up with the idea.
	
	CosmosUI URL:
	http://www.cosmosui.org/forums/viewtopic.php?t=257
	
	Bayta's Thread URL:
	http://www.worldofwarcraft.com/thread.aspx?fn=wow-interface-customization&t=7202
   ]]
   

-- Constants
LOOTDESTROYER_MAX_NUMBER_OF_BAGS = 4; -- add one because it starts at zero



-- Variables
LootDestroyer_Saved_LootFrameItem_OnClick = nil;
LootDestroyer_UnwantedItems = {};
LootDestroyer_Enabled = nil;

function LootDestroyer_OnLoad()
	LootDestroyer_Register();
end

function LootDestroyer_Register()
	if ( Cosmos_RegisterConfiguration ) then
		Cosmos_RegisterConfiguration(
			"COS_LOOTDESTROYER_HEADER",
			"SEPARATOR",
			LOOTDESTROYER_CONFIG_HEADER,
			LOOTDESTROYER_CONFIG_HEADER_INFO
		);
		Cosmos_RegisterConfiguration(
			"COS_LOOTDESTROYER_ENABLED",
			"CHECKBOX",
			LOOTDESTROYER_ENABLED,
			LOOTDESTROYER_ENABLED_INFO,
			LootDestroyer_Toggle_Enabled,
			0
		);
	else
		SlashCmdList["LOOTDESTROYERSLASHENABLE"] = LootDestroyer_Enable_ChatCommandHandler;
		SLASH_LOOTDESTROYERSLASHENABLE1 = "/lootdestroyerenable";
		SLASH_LOOTDESTROYERSLASHENABLE2 = "/ldenable";
		SLASH_LOOTDESTROYERSLASHENABLE3 = "/lootdestroyerdisable";
		SLASH_LOOTDESTROYERSLASHENABLE4 = "/lddisable";
		SlashCmdList["LOOTDESTROYERSLASHCLEAR"] = LootDestroyer_Clear_ChatCommandHandler;
		SLASH_LOOTDESTROYERSLASHCLEAR1 = "/lootdestroyerclear";
		SLASH_LOOTDESTROYERSLASHCLEAR2 = "/ldclear";
		SlashCmdList["LOOTDESTROYERSLASHSHOW"] = LootDestroyer_List_ChatCommandHandler;
		SLASH_LOOTDESTROYERSLASHSHOW1 = "/lootdestroyershow";
		SLASH_LOOTDESTROYERSLASHSHOW2 = "/ldshow";
		SLASH_LOOTDESTROYERSLASHSHOW3 = "/lootdestroyerlist";
		SLASH_LOOTDESTROYERSLASHSHOW4 = "/ldlist";
		SlashCmdList["LOOTDESTROYERSLASHTOGGLE"] = LootDestroyer_Toggle_ChatCommandHandler;
		SLASH_LOOTDESTROYERSLASHSHOW1 = "/lootdestroyertoggle";
		SLASH_LOOTDESTROYERSLASHSHOW2 = "/ldtoggle";
		SLASH_LOOTDESTROYERSLASHSHOW3 = "/lootdestroyeradd";
		SLASH_LOOTDESTROYERSLASHSHOW4 = "/ldadd";
		SLASH_LOOTDESTROYERSLASHSHOW5 = "/lootdestroyerremove";
		SLASH_LOOTDESTROYERSLASHSHOW6 = "/ldremove";
		this:RegisterEvent("VARIABLES_LOADED");
	end

	if ( Cosmos_RegisterChatCommand ) then
		local LootDestroyerEnableCommands = {"/lootdestroyerenable","/ldenable","/lootdestroyerdisable","/lddisable"};
		local LootDestroyerClearCommands = {"/lootdestroyerclear","/ldclear"};
		local LootDestroyerListCommands = {"/lootdestroyershow","/ldshow","/lootdestroyerlist","/ldlist"};
		local LootDestroyerToggleCommands = {"/lootdestroyertoggle","/ldtoggle","/lootdestroyeradd","/ldadd","/lootdestroyerremove","/ldremove"};
		Cosmos_RegisterChatCommand (
			"LOOTDESTROYER_ENABLE_COMMANDS", -- Some Unique Group ID
			LootDestroyerEnableCommands, -- The Commands
			LootDestroyer_Enable_ChatCommandHandler,
			LOOTDESTROYER_CHAT_COMMAND_ENABLE_INFO -- Description String
		);
		Cosmos_RegisterChatCommand (
			"LOOTDESTROYER_CLEAR_COMMANDS", -- Some Unique Group ID
			LootDestroyerClearCommands, -- The Commands
			LootDestroyer_Clear_ChatCommandHandler,
			LOOTDESTROYER_CHAT_COMMAND_CLEAR_INFO -- Description String
		);
		Cosmos_RegisterChatCommand (
			"LOOTDESTROYER_LIST_COMMANDS", -- Some Unique Group ID
			LootDestroyerListCommands, -- The Commands
			LootDestroyer_List_ChatCommandHandler,
			LOOTDESTROYER_CHAT_COMMAND_LIST_INFO -- Description String
		);
		Cosmos_RegisterChatCommand (
			"LOOTDESTROYER_TOGGLE_COMMANDS", -- Some Unique Group ID
			LootDestroyerToggleCommands, -- The Commands
			LootDestroyer_Toggle_ChatCommandHandler,
			LOOTDESTROYER_CHAT_COMMAND_TOGGLE_INFO -- Description String
		);
	end
end

function LootDestroyer_Enable_ChatCommandHandler(msg)
	msg = string.lower(msg);
	
	-- Toggle appropriately
	if ( (string.find(msg, 'on')) or ((string.find(msg, '1')) and (not string.find(msg, '-1')) ) ) then
		LootDestroyer_Toggle_Enabled(1);
	else
		if ( (string.find(msg, 'off')) or (string.find(msg, '0')) ) then
			LootDestroyer_Toggle_Enabled(0);
		else
			LootDestroyer_Toggle_Enabled(-1);
		end
	end
end

function LootDestroyer_Clear_ChatCommandHandler(msg)
	LootDestroyer_ClearUnwantedItems();
end

function LootDestroyer_List_ChatCommandHandler(msg)
	LootDestroyer_ShowUnwantedItems();
end

function LootDestroyer_Toggle_ChatCommandHandler(msg)
	LootDestroyer_ToggleUnwantedItem(msg);
end

function LootDestroyer_ClearUnwantedItems()
	LootDestroyer_UnwantedItems = {};
	LootDestroyer_Print(LOOTDESTROYER_CHAT_LIST_CLEARED);
end

function LootDestroyer_Setup_Hooks(toggle)
	if ( toggle = 1 ) then
		if ( ( LootFrameItem_OnClick ~= LootDestroyer_LootFrameItem_OnClick ) and (LootDestroyer_Saved_LootFrameItem_OnClick == nil) ) then
			LootDestroyer_Saved_LootFrameItem_OnClick = LootFrameItem_OnClick;
			LootFrameItem_OnClick = LootDestroyer_LootFrameItem_OnClick;
		end
	else
		if ( LootFrameItem_OnClick == LootDestroyer_LootFrameItem_OnClick) then
			LootFrameItem_OnClick = LootDestroyer_Saved_LootFrameItem_OnClick;
			LootDestroyer_Saved_LootFrameItem_OnClick = nil;
		end
	end
end


function LootDestroyer_OnEvent(event)
	if ( event == "VARIABLES_LOADED" ) then
		if(Cosmos_RegisterConfiguration == nil) then
			local value = getglobal("COS_LOOTDESTROYER_ENABLED_X");
			if (value == nil ) then
				-- defaults to off
				value = 0;
			end
			LootDestroyer_Toggle_Enabled(value);
		end
	end
end

function LootDestroyer_ShowUnwantedItems()
	LootDestroyer_Print(LOOTDESTROYER_CHAT_UNWANTED_ITEMS_LIST);
	for k,v in LootDestroyer_UnwantedItems do
		LootDestroyer_Print(v);
	end
end

function LootDestroyer_ToggleUnwantedItem(itemName)
	for k,v in LootDestroyer_UnwantedItems do
		if ( v == itemName) then
			table.remove(LootDestroyer_UnwantedItems, k);
			LootDestroyer_Print(format(LOOTDESTROYER_CHAT_ITEM_REMOVED, v));
			return;
		end
	end
	table.insert(LootDestroyer_UnwantedItems, itemName);
	LootDestroyer_Print(format(LOOTDESTROYER_CHAT_ITEM_ADDED, v));
end

function LootDestroyer_Toggle_Enabled(toggle)
	local oldvalue = LootDestroyer_Enabled;
	local newvalue = toggle;
	if ( ( toggle ~= 1 ) and ( toggle ~= 0 ) ) then
		if (oldvalue == 1) then
			newvalue = 0;
		elseif ( oldvalue == 0 ) then
			newvalue = 1;
		else
			newvalue = 0;
		end
	end
	LootDestroyer_Enabled = newvalue;
	setglobal("COS_LOOTDESTROYER_ENABLED_X", newvalue);
	LootDestroyer_Setup_Hooks(newvalue);
	if ( newvalue ~= oldvalue ) then
		if ( newvalue == 1 ) then
			LootDestroyer_Print(LOOTDESTROYER_CHAT_ENABLED);
		else
			LootDestroyer_Print(LOOTDESTROYER_CHAT_DISABLED);
		end
	end
	if ( Cosmos_RegisterConfiguration == nil ) then 
		RegisterForSave("COS_LOOTDESTROYER_ENABLED_X");	
	end
end

function LootDestroyer_IsUnwantedItem(itemName)
	for k,v in LootDestroyer_UnwantedItems do
		if ( v == itemName) then
			return 1;
		end
	end
	return 0;
end

function LootDestroyer_DestroyItem(itemName)
	if(CursorHasItem()) then
		LootDestroyer_Print(LOOTDESTROYER_CHAT_CURSORHASITEM);
		return false;
	end
	local bag = -1;
	local slot = -1;
	bag, slot = LootDestroyer_FindItemByName(itemName);
	if ( ( bag > -1 ) and ( slot > -1) ) then
		PickupContainerItem(bag, slot);
		DeleteCursorItem();
	end
end

function LootDestroyer_LootFrameItem_OnClick(button)
	if ( button == "LeftButton" ) then
		if ( IsShiftKeyDown() ~= true ) then
			if ( LootDestroyer_Enabled == 1) then
				local index = this:GetID();
				local itemName = getglobal("LootButton"..index.."Text"):GetText();
				if ( LootDestroyer_IsUnwantedItem(itemName) == 1 ) then
					LootDestroyer_DestroyItem(itemName);
				end
			end
		end
	end
	if ( button == "RightButton" ) then
		if ( IsShiftKeyDown() ~= true ) then
			local index = this:GetID();
			local itemName = getglobal("LootButton"..index.."Text"):GetText();
			LootDestroyer_ToggleUnwantedItem(itemName);
		end
	end
	if ( LootDestroyer_Saved_LootFrameItem_OnClick ) then
		LootDestroyer_Saved_LootFrameItem_OnClick(button);
	end
end
	
	
function LootDestroyer_Print(msg,r,g,b,frame,id,unknown4th)
	if(unknown4th) then
		local temp = id;
		id = unknown4th;
		unknown4th = id;
	end
				
	if (not r) then r = 1.0; end
	if (not g) then g = 1.0; end
	if (not b) then b = 1.0; end
	if ( frame ) then 
		frame:AddMessage(msg,r,g,b,id,unknown4th);
	else
		if ( DEFAULT_CHAT_FRAME ) then 
			DEFAULT_CHAT_FRAME:AddMessage(msg, r, g, b,id,unknown4th);
		end
	end
end

-- Item helper methods
function LootDestroyer_GetNumberOfBags()
	return LOOTDESTROYER_MAX_NUMBER_OF_BAGS;
end

function LootDestroyer_GetNumberOfBagSlots(bag)
	return GetContainerNumSlots(bag);
end


-- This function finds the specified item, return its bag, slot position
function LootDestroyer_FindItemByName(itemName)
	return LootDestroyer_FindItemByName_LastFirst(itemName);
end

-- This function starts at the end of the bags and goes backwards until it finds the item in question
function LootDestroyer_FindItemByName_LastFirst(itemName)
	local bag = 0;
	local slot = 0;
	for bag = LootDestroyer_GetNumberOfBags(), 0, -1 do
		for slot = LootDestroyer_GetNumberOfBagSlots(bag), 1, -1 do
			if ( LootDestroyer_ContainerItemHasName(itemName, bag, slot) == 1) then
				return bag, slot;
			end
		end
	end
	return -1, -1;
end


function LootDestroyer_ContainerItemHasName(itemName, bag, slot)
	local strings = LootDestroyerTooltip_GetItemInfoStrings(bag, slot);
	if ( ( strings[1] == itemName ) or ( strings[1].left == itemName ) or ( strings[1].right == itemName ) ) then
		return 1;
	else
		return 0;
	end
end

-- Helper methods from Cosmos
-- All credit should go to the Cosmos team for this one!

-- Gets all lines out of a tooltip.
function LootDestroyerTooltip_ScanTooltip(TooltipNameBase)
	if ( TooltipNameBase == nil ) then 
		TooltipNameBase = "LootDestroyerTooltip";
	end
	
	local strings = {};
	for idx = 1, 10 do
		local textLeft = nil;
		local textRight = nil;
		ttext = getglobal(TooltipNameBase.."TextLeft"..idx);
		if(ttext and ttext:IsVisible() and ttext:GetText() ~= nil)
		then
			textLeft = ttext:GetText();
		end
		ttext = getglobal(TooltipNameBase.."TextRight"..idx);
		if(ttext and ttext:IsVisible() and ttext:GetText() ~= nil)
		then
			textRight = ttext:GetText();
		end
		if (textLeft or textRight)
		then
			strings[idx] = {};
			strings[idx].left = textLeft;
			strings[idx].right = textRight;
		end
	end
	
	return strings;
end

-- Obtains all information about a bag/slot and returns it as an array 
function LootDestroyerTooltip_GetItemInfoStrings(bag,slot, TooltipNameBase)
	if ( TooltipNameBase == nil ) then 
		TooltipNameBase = "LootDestroyerTooltip";
	end

	--ClearTooltip(TooltipNameBase);

	local tooltip = getglobal(TooltipNameBase);
	
	-- Open tooltip & read contents
	tooltip:SetBagItem( bag, slot );
	local strings = LootDestroyerTooltip_ScanTooltip(TooltipNameBase);

	-- Done our duty, send report
	return strings;
end


