--[[
	SpellChecker2

	By sarf

	This mod allows you to auto-corrects messages, both the ones that you 
	send as well as the ones you receive. It also allows for experimental 
	on-the-fly correction.
	
	It is, in its core, an updated version of SpellChecker.

	Thanks goes to Bhaerau for suggesting it.
	Special thanks to Legorol and AnduinLothar for nagging me about on-the-fly correction,
	as well as about a binding for the people who may be less enthusiastic about spell 
	correctors than me.
	
	Check out SpellChecker2_Compatibility.lua for a list of addons that it is natively compatible with.
	
   ]]


SpellChecker2_Options = {
};

SpellChecker2_ChannelsToAvoid = {
};

SpellChecker2_User_Corrections = {
};

-- session-specific variables
SpellChecker2_Setup = {
};

function SpellChecker2_OnLoad()
	this:RegisterEvent("VARIABLES_LOADED");
	this:RegisterEvent("ADDON_LOADED");

	SpellChecker2_Saved_ChatEdit_ParseText = ChatEdit_ParseText;
	ChatEdit_ParseText = SpellChecker2_ChatEdit_ParseText;
end	

function SpellChecker2_OnEvent()
	if ( event == "VARIABLES_LOADED" ) then
		SpellChecker2_VariablesLoaded();
	elseif( event == "ADDON_LOADED" ) then
		SpellChecker2_CompatibilityCheck();
	end
end	

function SpellChecker2_VariablesLoaded()
	if ( not SpellChecker2_Options ) then
		SpellChecker2_Options = {};
	end
	for k, v in pairs(SPELLCHECKER2_OPTIONS_DEFAULT) do
		if ( SpellChecker2_Options[k] == nil ) then
			SpellChecker2_Options[k] = v;
		end
	end
	local cmdName = "SPELLCHECKER2";
	SlashCmdList[cmdName] = SpellChecker2_SlashCommand;
	for k, v in pairs(SPELLCHECKER2_SLASH_COMMANDS) do
		setglobal("SLASH_"..cmdName..k, v);
	end
	SpellChecker2_CompatibilityCheck();

-- Hooking functions

	SpellChecker2_Saved_SendChatMessage = SendChatMessage;
	SendChatMessage = SpellChecker2_SendChatMessage;
	
	SpellChecker2_Saved_ChatFrame_OnEvent = ChatFrame_OnEvent;
	ChatFrame_OnEvent = SpellChecker2_ChatFrame_OnEvent;

-- Options

	SpellChecker2_Options_VariablesLoaded();
	
end

function SpellChecker2_GetNextArgument(msg)
	local index;
	index = string.find(msg, " ");
	if ( index ) then
		return strsub(msg, 1, index-1), strsub(msg, index+1);
	else
		return msg, "";
	end
end

function SpellChecker2_SlashCommand(msg)
	local command, params = SpellChecker2_GetNextArgument(msg);
	for k, v in pairs(SPELLCHECKER2_SLASH_GUI) do
		if ( command == v ) then
			SpellChecker2_SlashCommand_GUI();
			return;
		end
	end
	for k, v in pairs(SPELLCHECKER2_SLASH_ADD) do
		if ( command == v ) then
			SpellChecker2_SlashCommand_AddCorrection(SpellChecker2_GetNextArgument(params));
			return;
		end
	end
	for k, v in pairs(SPELLCHECKER2_SLASH_REMOVE) do
		if ( command == v ) then
			SpellChecker2_SlashCommand_RemoveCorrection(SpellChecker2_GetNextArgument(params));
			return;
		end
	end
	SpellChecker2_SlashCommand_ShowUsage();
end

function SpellChecker2_SlashCommand_ShowUsage()
	for k, v in pairs(SPELLCHECKER2_SLASH_USAGE) do
		SpellChecker2_ShowLine(v);
	end
end

function SpellChecker2_SlashCommand_AddCorrection(thingToCorrect, correctThing)
	if ( thingToCorrect ) and ( strlen(thingToCorrect) > 0 ) 
		and ( correctThing ) and ( strlen(correctThing) > 0 ) then
		SpellChecker2_AddCorrection(thingToCorrect, correctThing);
		SpellChecker2_ShowLine(format(SPELLCHECKER2_MSG_ADD_CORRECTION, thingToCorrect, correctThing));
	else
		SpellChecker2_ShowLine(SPELLCHECKER2_MSG_EMPTY_CORRECTION);
	end
end

function SpellChecker2_SlashCommand_RemoveCorrection(thingToCorrect)
	if ( thingToCorrect ) and ( strlen(thingToCorrect) > 0 ) then
		SpellChecker2_RemoveCorrection(thingToCorrect);
		SpellChecker2_ShowLine(format(SPELLCHECKER2_MSG_REMOVE_CORRECTION, thingToCorrect));
	else
		SpellChecker2_ShowLine(SPELLCHECKER2_MSG_EMPTY_CORRECTION);
	end
end

function SpellChecker2_SlashCommand_GUI()
	if ( not IsAddOnLoaded("SpellChecker2GUI") ) then
		LoadAddOn("SpellChecker2GUI");
	end
	if ( SpellChecker2_GUI_ToggleShow ) then
		SpellChecker2_GUI_ToggleShow();
	else
		SpellChecker2_ShowLine(SPELLCHECKER2_MSG_NYI);
	end
end

function SpellChecker2_ShowLine(line)
	if ( ChatFrame1 ) and ( ChatFrame1.AddMessage ) then
		ChatFrame1:AddMessage(line, 1, 1, 0.1);
	end
end

function SpellChecker2_AddCorrection(thingToCorrect, correctThing)
	local wl = SpellChecker2_WordList[thingToCorrect];
	if ( wl ) and ( wl == correctThing ) then
		SpellChecker2_User_Corrections[thingToCorrect] = nil;
	else
		if ( thingToCorrect ) and ( strlen(thingToCorrect) > 0 ) 
			and ( correctThing ) and ( strlen(correctThing) > 0 ) then
			SpellChecker2_User_Corrections[thingToCorrect] = correctThing;
		end
	end
end

function SpellChecker2_RemoveCorrection(thingToCorrect)
	if ( thingToCorrect ) and ( strlen(thingToCorrect) > 0 ) then
		if ( SpellChecker2_WordList[thingToCorrect] ) then
			SpellChecker2_User_Corrections[thingToCorrect] = thingToCorrect;
		else
			SpellChecker2_User_Corrections[thingToCorrect] = nil;
		end
	end
end

SpellChecker2_CorrectionStr = "";

function SpellChecker2_CorrectionFunc(str, sep)
	local user = SpellChecker2_User_Corrections[str];
	local wl = nil;
	if ( not user ) then
		wl = SpellChecker2_WordList[str];
	end
	if ( user ) then
		str = user;
	elseif ( wl ) then
		str = wl;
	end
	if ( sep ) then
		SpellChecker2_CorrectionStr = SpellChecker2_CorrectionStr..str..sep;
	elseif ( str ) and ( str ~= "" ) then
		SpellChecker2_CorrectionStr = SpellChecker2_CorrectionStr..str;
	end
end

function SpellChecker2_CorrectMessage(msg, noStartCapitalization)
	local firstChar = strsub(msg, 1, 1);
	-- my attempt to be compatible with most addons sending info.
	if ( firstChar == "<" ) then
		return msg;
	end
	SpellChecker2_CorrectionStr = "";
	SpellChecker2_CorrectionFunc((string.gsub(msg, SPELLCHECKER2_CORRECTIONEXPRESSION, SpellChecker2_CorrectionFunc)))
	if ( not noStartCapitalization ) and (SpellChecker2_Options.startCapitalization ) then
		SpellChecker2_CorrectionStr = strupper(strsub(SpellChecker2_CorrectionStr, 1, 1))..strsub(SpellChecker2_CorrectionStr, 2);
	end
	return SpellChecker2_CorrectionStr;
end

function SpellChecker2_MatchChannelsInList(channel, number, channelsList)
	if ( not channel ) then
		return false;
	end
	if ( not channelsList ) then
		return false;
	end
	if ( channel == "CHANNEL" ) then
		if ( not number ) then
			return false;
		end
		channel = channel..number;
	end
	for k, v in pairs(channelsList) do
		if ( v == channel ) then
			return true;
		end
	end
	return false;
end


function SpellChecker2_SendChatMessage(msg, msgType, language, number, p1, p2, p3, p4, p5)
	if ( SpellChecker2_Setup.ignoreNextMessage ) 
		or ( not SpellChecker2_Options.enabled ) 
		or ( not SpellChecker2_Options.correctOutgoingMessages ) then
		SpellChecker2_Setup.ignoreNextMessage = false;
	elseif ( msgType ) then
		local correct = false;
		correct = SpellChecker2_MatchChannelsInList(msgType, number, SpellChecker2_Options.channelsToCorrect);

		if ( correct ) then
			if ( SpellChecker2_MatchChannelsInList(msgType, number, SpellChecker2_ChannelsToAvoid) ) then
				correct = false;
			end
		end
		if ( correct ) then
			msg = SpellChecker2_CorrectMessage(msg, (msgType=="EMOTE"));
		end
	end
	return SpellChecker2_Saved_SendChatMessage(msg, msgType, language, number, p1, p2, p3, p4, p5);
end

function SpellChecker2_GetEventList()
	if ( SpellChecker2_Setup.eventListDirty ) or ( not SpellChecker2_Setup.eventList ) then
		if ( not SpellChecker2_Options ) or ( not SpellChecker2_Options.channelsToCorrect ) then
			return nil;
		end
		if ( not SpellChecker2_Setup.eventList ) then
			SpellChecker2_Setup.eventList = {};
		else
			for k, v in SpellChecker2_Setup.eventList do
				SpellChecker2_Setup.eventList[k] = false;
			end
		end
		local ctg = nil;
		for k, v in SpellChecker2_Options.channelsToCorrect do
			ctg = ChatTypeGroup[v];
			if ( ctg ) then
				for key, event in ctg do
					SpellChecker2_Setup.eventList[event] = true;
				end
			end
		end
	end
	return SpellChecker2_Setup.eventList;
end

function SpellChecker2_ChatFrame_OnEvent(event)
	local correct = true;
	if ( not SpellChecker2_Options.ignoranceIsBliss ) then
		if ( arg2 == UnitName("player") ) then
			correct = false;
		end
	end
	if ( not SpellChecker2_Options.enabled ) or ( not SpellChecker2_Options.correctIncomingMessages ) then
		correct = false;
	end
	if ( correct ) then
		correct = false;
		local eventList = SpellChecker2_GetEventList();
		if ( eventList[event] ) then
			correct = true;
		elseif ( event == "CHAT_MSG_CHANNEL" ) then
			correct = SpellChecker2_MatchChannelsInList("CHANNEL", tonumber(arg8), SpellChecker2_Options.channelsToCorrect);
	
			if ( correct ) then
				if ( SpellChecker2_MatchChannelsInList("CHANNEL", tonumber(arg8), SpellChecker2_ChannelsToAvoid) ) then
					correct = false;
				end
			end
		end
		if ( correct ) then
			arg1 = SpellChecker2_CorrectMessage(arg1, true);
		end
	end
	return SpellChecker2_Saved_ChatFrame_OnEvent(event);
end



function SpellChecker2_ChatEdit_ParseText(editBox, send, p1, p2, p3, p4, p5)
	local text = editBox:GetText();
	local shouldCorrect = SpellChecker2_Options.onTheFlyCorrection;

	if ( not text ) or ( strlen(text) <= 0 ) then
		shouldCorrect = false;
	end
	if ( shouldCorrect ) then
		local slashCmd = "SLASH_SCRIPT";
		local i = 1;
		local cmd = getglobal(slashCmd..i);
		while ( cmd ) do
			if ( strsub(text, 1, strlen(cmd)) == cmd ) then
				shouldCorrect = false;
				break;
			end
			i = i + 1;
			cmd = getglobal(slashCmd..i)
		end
	end
	if ( shouldCorrect ) then
		slashCmd = "SLASH_SPELLCHECKER2";
		i = 1;
		cmd = getglobal(slashCmd..i);
		while ( cmd ) do
			if ( strsub(text, 1, strlen(cmd)) == cmd ) then
				shouldCorrect = false;
				break;
			end
			i = i + 1;
			cmd = getglobal(slashCmd..i)
		end
	end
	
	if ( shouldCorrect ) then
		text = SpellChecker2_CorrectMessage(text, true);
		editBox:SetText(text);
	end

	return SpellChecker2_Saved_ChatEdit_ParseText(editBox, send, p1, p2, p3, p4, p5);
end
