--[[
	Weapon Buttons

	By sarf

	This mod gives you a draggable window with the mainhand / offhand buttons. 
	This way you don't have to open the paperdoll to equip weapons anymore.

	Thanks goes to 
	
	CosmosUI URL:
	http://www.cosmosui.org/forums/viewtopic.php?t=
	
   ]]


-- Constants

-- Variables
WeaponButtons_Enabled = 0;
WeaponButtons_DisplayTooltips = 1;
WeaponButtons_LockPosition = 0;

WeaponButtons_Cosmos_Registered = 0;

-- executed on load, calls general set-up functions
function WeaponButtons_OnLoad()
	WeaponButtons_Register();
end

-- registers the mod with Cosmos
function WeaponButtons_Register_Cosmos()
	if ( ( Cosmos_RegisterConfiguration ) and ( WeaponButtons_Cosmos_Registered == 0 ) ) then
		Cosmos_RegisterConfiguration(
			"COS_WEAPONBUTTONS",
			"SECTION",
			TEXT(WEAPONBUTTONS_CONFIG_HEADER),
			TEXT(WEAPONBUTTONS_CONFIG_HEADER_INFO)
		);
		Cosmos_RegisterConfiguration(
			"COS_WEAPONBUTTONS_HEADER",
			"SEPARATOR",
			TEXT(WEAPONBUTTONS_CONFIG_HEADER),
			TEXT(WEAPONBUTTONS_CONFIG_HEADER_INFO)
		);
		Cosmos_RegisterConfiguration(
			"COS_WEAPONBUTTONS_ENABLED",
			"CHECKBOX",
			TEXT(WEAPONBUTTONS_ENABLED),
			TEXT(WEAPONBUTTONS_ENABLED_INFO),
			WeaponButtons_Toggle_Enabled,
			WeaponButtons_Enabled
		);
		Cosmos_RegisterConfiguration(
			"COS_WEAPONBUTTONS_DISPLAY_TOOLTIPS",
			"CHECKBOX",
			TEXT(WEAPONBUTTONS_DISPLAY_TOOLTIPS),
			TEXT(WEAPONBUTTONS_DISPLAY_TOOLTIPS_INFO),
			WeaponButtons_Toggle_DisplayTooltips,
			WeaponButtons_DisplayTooltips
		);
		Cosmos_RegisterConfiguration(
			"COS_WEAPONBUTTONS_LOCK_POSITION",
			"CHECKBOX",
			TEXT(WEAPONBUTTONS_LOCK_POSITION),
			TEXT(WEAPONBUTTONS_LOCK_POSITION_INFO),
			WeaponButtons_Toggle_LockPosition,
			WeaponButtons_LockPosition
		);
		if ( Cosmos_RegisterChatCommand ) then
			local WeaponButtonsMainCommands = {"/weaponbutton","/weaponbuttons","/wb"};
			Cosmos_RegisterChatCommand (
				"WEAPONBUTTONS_MAIN_COMMANDS", -- Some Unique Group ID
				WeaponButtonsMainCommands, -- The Commands
				WeaponButtons_Main_ChatCommandHandler,
				WEAPONBUTTONS_CHAT_COMMAND_INFO -- Description String
			);
		end
		WeaponButtons_Cosmos_Registered = 1;
	end
end

-- registers the mod with the system, integrating it with slash commands and "master" AddOns
function WeaponButtons_Register()
	if ( Cosmos_RegisterConfiguration ) then
		WeaponButtons_Register_Cosmos();
	else
		SlashCmdList["WEAPONBUTTONSSLASHENABLE"] = WeaponButtons_Main_ChatCommandHandler;
		SLASH_WEAPONBUTTONSSLASHENABLE1 = "/weaponbuttons";
		SLASH_WEAPONBUTTONSSLASHENABLE2 = "/weaponbutton";
		SLASH_WEAPONBUTTONSSLASHENABLE3 = "/wb";
		this:RegisterEvent("VARIABLES_LOADED");
	end

end

function WeaponButtons_Extract_NextParameter(msg)
	local params = msg;
	local command = params;
	local index = strfind(command, " ");
	if ( index ) then
		command = strsub(command, 1, index-1);
		params = strsub(params, index+1);
	else
		params = "";
	end
	return command, params;
end



-- Handles chat - e.g. slashcommands - enabling/disabling the WeaponButtons
function WeaponButtons_Main_ChatCommandHandler(msg)
	
	local func = WeaponButtons_Toggle_Enabled;
	
	local toggleFunc = true;
	
	if ( ( not msg) or ( strlen(msg) <= 0 ) ) then
		WeaponButtons_Print(WEAPONBUTTONS_CHAT_COMMAND_USAGE);
		return;
	end
	
	local commandName, params = WeaponButtons_Extract_NextParameter(msg);
	
	if ( ( commandName ) and ( strlen(commandName) > 0 ) ) then
		commandName = string.lower(commandName);
	else
		commandName = "";
	end
	
	if ( strfind( commandName, "tooltip" ) ) then
		func = WeaponButtons_Toggle_DisplayTooltips;
	elseif ( strfind( commandName, "lock" ) ) then
		func = WeaponButtons_Toggle_LockPosition;
	elseif ( ( strfind( commandName, "help" ) ) or ( strfind( commandName, "/?" ) ) or ( strfind( commandName, "-?" ) ) ) then
		WeaponButtons_Print(WEAPONBUTTONS_CHAT_COMMAND_USAGE);
		return;
	elseif ( ( not params ) or ( strlen(params) <= 0 ) ) then
		-- defaults to show command
		params = commandName;
	end
	
	if ( toggleFunc ) then
		-- Toggle appropriately
		if ( (string.find(params, 'on')) or ((string.find(params, '1')) and (not string.find(params, '-1')) ) ) then
			func(1);
		else
			if ( (string.find(params, 'off')) or (string.find(params, '0')) ) then
				func(0);
			else
				func(-1);
			end
		end
	else
		func();
	end
end

-- Handles chat - e.g. slashcommands - enabling/disabling the WeaponButtons
function WeaponButtons_Enable_ChatCommandHandler(msg)
	msg = string.lower(msg);
	
	-- Toggle appropriately
	if ( (string.find(msg, 'on')) or ((string.find(msg, '1')) and (not string.find(msg, '-1')) ) ) then
		WeaponButtons_Toggle_Enabled(1);
	else
		if ( (string.find(msg, 'off')) or (string.find(msg, '0')) ) then
			WeaponButtons_Toggle_Enabled(0);
		else
			WeaponButtons_Toggle_Enabled(-1);
		end
	end
end

-- Handles chat - e.g. slashcommands - enabling/disabling the tooltips of the WeaponButtons
function WeaponButtons_DisplayTooltips_ChatCommandHandler(msg)
	msg = string.lower(msg);
	
	-- Toggle appropriately
	if ( (string.find(msg, 'on')) or ((string.find(msg, '1')) and (not string.find(msg, '-1')) ) ) then
		WeaponButtons_Toggle_DisplayTooltips(1);
	else
		if ( (string.find(msg, 'off')) or (string.find(msg, '0')) ) then
			WeaponButtons_Toggle_DisplayTooltips(0);
		else
			WeaponButtons_Toggle_DisplayTooltips(-1);
		end
	end
end

-- Handles chat - e.g. slashcommands - locking/unlocking the position of the WeaponButtons
function WeaponButtons_LockPosition_ChatCommandHandler(msg)
	msg = string.lower(msg);
	
	-- Toggle appropriately
	if ( (string.find(msg, 'on')) or ((string.find(msg, '1')) and (not string.find(msg, '-1')) ) ) then
		WeaponButtons_Toggle_LockPosition(1);
	else
		if ( (string.find(msg, 'off')) or (string.find(msg, '0')) ) then
			WeaponButtons_Toggle_LockPosition(0);
		else
			WeaponButtons_Toggle_LockPosition(-1);
		end
	end
end

-- Handles events
function WeaponButtons_OnEvent(event)
	if ( event == "VARIABLES_LOADED" ) then
		if ( WeaponButtons_Cosmos_Registered == 0 ) then
			local value = getglobal("COS_WEAPONBUTTONS_ENABLED_X");
			if (value == nil ) then
				-- defaults to off
				value = 0;
			end
			WeaponButtons_Toggle_Enabled(value);
			value = getglobal("COS_WEAPONBUTTONS_DISPLAY_TOOLTIPS_X");
			if (value == nil ) then
				-- defaults to on
				value = 1;
			end
			WeaponButtons_Toggle_DisplayTooltips(value);
			value = getglobal("COS_WEAPONBUTTONS_LOCK_POSITION_X");
			if (value == nil ) then
				-- defaults to off
				value = 0;
			end
			WeaponButtons_Toggle_LockPosition(value);
		end
	end
end

-- Toggles the enabled/disabled state of an option and returns the new state
--  if toggle is 1, it's enabled
--  if toggle is 0, it's disabled
--   otherwise, it's toggled
function WeaponButtons_Generic_Toggle(toggle, variableName, CVarName, enableMessage, disableMessage, CosmosVarName)
	local oldvalue = getglobal(variableName);
	local newvalue = toggle;
	if ( ( toggle ~= 1 ) and ( toggle ~= 0 ) ) then
		if (oldvalue == 1) then
			newvalue = 0;
		elseif ( oldvalue == 0 ) then
			newvalue = 1;
		else
			newvalue = 0;
		end
	end
	setglobal(variableName, newvalue);
	setglobal(CVarName, newvalue);
	if ( ( newvalue ~= oldvalue ) and ( TellTrack_Cosmos_Registered == 0 ) ) then
		if ( newvalue == 1 ) then
			WeaponButtons_Print(TEXT(getglobal(enableMessage)));
		else
			WeaponButtons_Print(TEXT(getglobal(disableMessage)));
		end
	end
	WeaponButtons_Register_Cosmos();
	if ( WeaponButtons_Cosmos_Registered == 0 ) then 
		RegisterForSave(variableName);
		RegisterForSave(CVarName);
	else
		local varName = CosmosVarName;
		if ( not varName ) then
			varName = strsub(CVarName, 1, strlen(CVarName)-2);
		end
		Cosmos_UpdateValue(varName, CSM_CHECKONOFF, newvalue);
		Cosmos_SetCVar(CVarName, newvalue);
	end
	return newvalue;
end

-- Toggles the enabled/disabled state of the WeaponButtons
--  if toggle is 1, it's enabled
--  if toggle is 0, it's disabled
--   otherwise, it's toggled
function WeaponButtons_Toggle_Enabled(toggle)
	local newvalue = WeaponButtons_Generic_Toggle(toggle, "WeaponButtons_Enabled", "COS_WEAPONBUTTONS_ENABLED_X", "WEAPONBUTTONS_CHAT_ENABLED", "WEAPONBUTTONS_CHAT_DISABLED");
	if ( newvalue == 1 ) then
		WeaponButtonsFrame:Show();
	else
		WeaponButtonsFrame:Hide();
	end
end

-- Toggles whether tooltips are displayed on mousing over of the WeaponButtons
--  if toggle is 1, they are displayed
--  if toggle is 0, they are not displayed
--   otherwise, they are toggled
function WeaponButtons_Toggle_DisplayTooltips(toggle)
	local newvalue = WeaponButtons_Generic_Toggle(toggle, "WeaponButtons_DisplayTooltips", "COS_WEAPONBUTTONS_DISPLAY_TOOLTIPS_X", "WEAPONBUTTONS_CHAT_DISPLAY_TOOLTIPS_ENABLED", "WEAPONBUTTONS_CHAT_DISPLAY_TOOLTIPS_DISABLED");
end

-- Toggles whether the window is locked to its current position or not.
--  if toggle is 1, they are displayed
--  if toggle is 0, they are not displayed
--   otherwise, they are toggled
function WeaponButtons_Toggle_LockPosition(toggle)
	local newvalue = WeaponButtons_Generic_Toggle(toggle, "WeaponButtons_LockPosition", "COS_WEAPONBUTTONS_LOCK_POSITION_X", "WEAPONBUTTONS_CHAT_LOCK_POSITION_ENABLED", "WEAPONBUTTONS_CHAT_LOCK_POSITION_DISABLED");
end

-- Prints out text to a chat box.
function WeaponButtons_Print(msg,r,g,b,frame,id,unknown4th)
	if(unknown4th) then
		local temp = id;
		id = unknown4th;
		unknown4th = id;
	end
				
	if (not r) then r = 1.0; end
	if (not g) then g = 1.0; end
	if (not b) then b = 1.0; end
	if ( frame ) then 
		frame:AddMessage(msg,r,g,b,id,unknown4th);
	else
		if ( DEFAULT_CHAT_FRAME ) then 
			DEFAULT_CHAT_FRAME:AddMessage(msg, r, g, b,id,unknown4th);
		end
	end
end

function WeaponButton_GetInventoryInfoSlotName(slotName)
	local inventorySlotInfoName = slotName;
	local charIndex = strfind(inventorySlotInfoName, "Character");
	if ( charIndex ) then inventorySlotInfoName = strsub(inventorySlotInfoName, charIndex); end
	inventorySlotInfoName = strsub(inventorySlotInfoName,10);
	return inventorySlotInfoName;
end

function WeaponButtonPaperDollItemSlotButton_OnLoad()
	this:RegisterEvent("UNIT_INVENTORY_CHANGED");
	this:RegisterEvent("ITEM_LOCK_CHANGED");
	this:RegisterEvent("CURSOR_UPDATE");
	this:RegisterEvent("BAG_UPDATE_COOLDOWN");
	this:RegisterEvent("SHOW_COMPARE_TOOLTIP");
	this:RegisterEvent("UPDATE_INVENTORY_ALERTS");
	this:RegisterForDrag("LeftButton");
	this:RegisterForClicks("LeftButtonUp", "RightButtonUp");
	local slotName = this:GetName();
	local id;
	local textureName;

	id, textureName = GetInventorySlotInfo(WeaponButton_GetInventoryInfoSlotName(slotName));
	
	this:SetID(id);
	local texture = getglobal(slotName.."IconTexture");
	texture:SetTexture(textureName);
	this.backgroundTextureName = textureName;
	PaperDollItemSlotButton_Update();
end


function WeaponButtonPaperDollItemSlotButton_OnEnter()
	if ( WeaponButtons_DisplayTooltips == 1 ) then
		GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
		local hasItem, hasCooldown, repairCost = GameTooltip:SetInventoryItem("player", this:GetID());
	    if ( not hasItem ) then
			GameTooltip:SetText(TEXT(getglobal(strupper(WeaponButton_GetInventoryInfoSlotName(this:GetName())))));
		end
		if ( hasCooldown ) then
			this.updateTooltip = TOOLTIP_UPDATE_TIME;
		else
			this.updateTooltip = nil;
		end
	--	if ( MerchantFrame:IsVisible() ) then
	--		ShowInventorySellCursor(this:GetID());
	--	end
		if ( InRepairMode() and repairCost and (repairCost > 0) ) then
			GameTooltip:AddLine(TEXT(REPAIR_COST), "", 1, 1, 1);
			SetTooltipMoney(GameTooltip, repairCost);
			GameTooltip:Show();
		end
	end
end

